const mysql = require('mysql2/promise');
const { MySQLAdapter } = require('../utils/mysqlAdapter');
const { InMemoryAdapter } = require('../utils/inMemoryAdapter');

let pool;
let db;

async function ensureDatabaseExists({ host, port, user, password, database }) {

    const tempPool = mysql.createPool({
        host,
        port,
        user,
        password,
        waitForConnections: true,
        connectionLimit: 5,
        queueLimit: 0,
    });
    try {
        await tempPool.query(`CREATE DATABASE IF NOT EXISTS \`${database}\``);
        console.log(`Database '${database}' ensured`);
    } catch (err) {
        console.error('Failed to ensure database:', err);
        throw err;
    } finally {
        await tempPool.end();
    }
}

async function connectToMySQL() {
    try {
        const url = process.env.MYSQL_URL || process.env.DATABASE_URL;
        const host = process.env.MYSQL_HOST;
        const port = Number(process.env.MYSQL_PORT);
        const user = process.env.MYSQL_USER;
        const password = process.env.MYSQL_PASSWORD;
        const database = process.env.MYSQL_DATABASE || process.env.MYSQL_DB_NAME;

        console.log('Connecting to MySQL...');

        const skipCreate = String(
            process.env.MYSQL_SKIP_CREATE_DB ||
            process.env.SKIP_DB_CREATE ||
            (process.env.NODE_ENV === 'production' ? 'true' : 'false')
        ).toLowerCase() === 'true';

        if (!skipCreate) {
            await ensureDatabaseExists({ host, port, user, password, database });
        } else {
            console.log(
                `Skipping database creation check for '${database}' (MYSQL_SKIP_CREATE_DB=${process.env.MYSQL_SKIP_CREATE_DB || 'unset'})`
            );
        }

        if (url) {
            pool = mysql.createPool(url);
        } else {
            pool = mysql.createPool({
                host,
                port,
                user,
                password,
                database,
                waitForConnections: true,
                connectionLimit: 10,
                queueLimit: 0,
            });
        }

        await pool.query('SELECT 1');
        db = new MySQLAdapter(pool);
        console.log('✅ Connected to MySQL successfully');

        await createTablesIfNotExist();
        await createIndexes();
        return db;
    } catch (error) {
        console.error('MySQL connection error:', error);
        const defaultFallback = (process.env.NODE_ENV === 'production') ? 'false' : 'true';
        const allowFallback = String(process.env.ALLOW_DEV_MEMORY_FALLBACK || defaultFallback).toLowerCase() !== 'false';
        if (allowFallback) {
            console.warn('⚠️ Falling back to in-memory database. Data will NOT persist.');
            console.warn('   Set ALLOW_DEV_MEMORY_FALLBACK=false and configure MySQL env to persist.');
            db = new InMemoryAdapter();
            pool = null;
            return db;
        }
        throw error;
    }
}

async function createTablesIfNotExist() {
    if (!pool) {
        console.log('Skipping table creation in in-memory mode');
        return;
    }
    const statements = [
        `CREATE TABLE IF NOT EXISTS \`users\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`news\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`categories\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`roles\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`comments\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`likes\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`ads\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`settings\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`breaking_news\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`company_pages\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`gold_silver_rates\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`careers\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`contact_messages\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`,
        `CREATE TABLE IF NOT EXISTS \`subscribers\` (id VARCHAR(36) PRIMARY KEY, data JSON, created_at DATETIME, updated_at DATETIME)`
    ];
    for (const s of statements) {
        await pool.query(s);
    }
    console.log('✅ MySQL tables ensured');
}

function collectExpectedIndexSpecs(jsonIndexSupported) {
    let indexes = [];
    if (jsonIndexSupported) {
        indexes = [
            { table: 'users', indexName: 'idx_users_email', expr: "((CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$.email')) AS CHAR(191))))", field: 'email' },
            { table: 'users', indexName: 'idx_users_role', expr: "((CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$.role')) AS CHAR(191))))", field: 'role' },
            { table: 'news', indexName: 'idx_news_slug', expr: "((CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$.slug')) AS CHAR(191))))", field: 'slug' },
            { table: 'news', indexName: 'idx_news_status', expr: "((CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$.status')) AS CHAR(50))))", field: 'status' },
            { table: 'news', indexName: 'idx_news_category', expr: "((CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$.category')) AS CHAR(50))))", field: 'category' },
            { table: 'news', indexName: 'idx_news_published_at', expr: "((CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$.published_at')) AS DATETIME)))", field: 'published_at' },
            { table: 'categories', indexName: 'idx_categories_slug', expr: "((CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$.slug')) AS CHAR(191))))", field: 'slug' }
        ];
    } else {
        indexes = [
            { table: 'users', indexName: 'idx_users_created_at', expr: 'created_at' },
            { table: 'news', indexName: 'idx_news_created_at', expr: 'created_at' }
        ];
    }
    return indexes;
}

async function createIndexes() {
    if (!pool) {
        console.log('Skipping index creation in in-memory mode');
        return;
    }
    let jsonIndexSupported = true;
    try {
        const [verRows] = await pool.query('SELECT VERSION() AS version');
        const versionStr = (verRows && verRows[0] && verRows[0].version) ? String(verRows[0].version) : '';
        const isMariaDB = /MariaDB/i.test(versionStr);
        jsonIndexSupported = !isMariaDB;
    } catch (verErr) {
        jsonIndexSupported = false;
    }
    async function ensureIndex({ table, indexName, expr, unique = false }) {
        try {
            const [rows] = await pool.query(
                `SELECT COUNT(*) AS cnt FROM information_schema.statistics WHERE table_schema = DATABASE() AND table_name = ? AND index_name = ?`,
                [table, indexName]
            );
            const exists = rows && rows[0] && Number(rows[0].cnt) > 0;
            if (exists) {
                return false;
            }
            const sql = `${unique ? 'CREATE UNIQUE INDEX' : 'CREATE INDEX'} ${indexName} ON \`${table}\` (${expr})`;
            await pool.query(sql);
            console.log(`✅ Created index ${indexName} on ${table}`);
            return true;
        } catch (err) {
            console.warn(` Index '${indexName}' on '${table}' warning:`, err.message);
            return false;
        }
    }

    const indexes = collectExpectedIndexSpecs(jsonIndexSupported);

    for (const spec of indexes) {
        await ensureIndex(spec);
    }
    console.log('✅ MySQL indexes ensured');
}

function getDB() {
    if (!db) {
        throw new Error('Database not connected. Call connectToMySQL() first.');
    }
    return db;
}

function getPool() {
    if (!pool) {
        throw new Error('Database pool not connected. Call connectToMySQL() first.');
    }
    return pool;
}

module.exports = {
    connectToMySQL,
    getDB,
    getPool,
    createIndexes
};
