const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const { getDB } = require('../config/db');

const JWT_SECRET = process.env.JWT_SECRET || 'your-secret-key-change-this-in-production';
const JWT_EXPIRES_IN = process.env.JWT_EXPIRES_IN || '7d';


const register = async (req, res) => {
    try {
        const { email, password, name, role = 'super-admin' } = req.body;

        
        if (role !== 'super-admin') {
            return res.status(403).json({
                success: false,
                message: 'Only super-admin accounts can be registered.'
            });
        }

        
        if (!email || !password || !name) {
            return res.status(400).json({
                success: false,
                message: 'Email, password, and name are required'
            });
        }

        const db = getDB();
        const usersColl = db.collection('users');

        
        const existingUsers = await usersColl.find({ email }).toArray();
        if (existingUsers && existingUsers.length > 0) {
            return res.status(409).json({
                success: false,
                message: 'User with this email already exists'
            });
        }

        
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(password, salt);

        
        
        const userData = {
            email,
            password: hashedPassword,
            name,
            role,
            isActive: true
        };

        const result = await usersColl.insertOne(userData);

        
        const newUserId = result.insertedId;
        const responseUser = {
            id: newUserId,
            ...userData,
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString()
        };

        
        delete responseUser.password;

        res.status(201).json({
            success: true,
            message: 'User registered successfully',
            data: responseUser
        });
    } catch (error) {
        console.error('Register error:', error);
        res.status(500).json({
            success: false,
            message: 'Error registering user',
            error: error.message
        });
    }
};


const login = async (req, res) => {
    try {
        const { email, password } = req.body;

        
        if (!email || !password) {
            return res.status(400).json({
                success: false,
                message: 'Email and password are required'
            });
        }

        const db = getDB();
        const usersColl = db.collection('users');

        
        
        const users = await usersColl.find({ email }).toArray();

        if (!users || users.length === 0) {
            return res.status(401).json({
                success: false,
                message: 'Invalid credentials'
            });
        }

        const user = users[0];

        
        if (user.role !== 'super-admin') {
            return res.status(403).json({
                success: false,
                message: 'Access Denied: Only Super-Admins can access this portal.'
            });
        }

        
        if (user.isActive === false) { 
            return res.status(403).json({
                success: false,
                message: 'Account is deactivated'
            });
        }

        
        const isPasswordValid = await bcrypt.compare(password, user.password);
        if (!isPasswordValid) {
            return res.status(401).json({
                success: false,
                message: 'Invalid credentials'
            });
        }

        
        const token = jwt.sign(
            {
                userId: user.id || user._id, 
                email: user.email,
                role: user.role,
                name: user.name
            },
            JWT_SECRET,
            { expiresIn: JWT_EXPIRES_IN }
        );

        
        const userResponse = { ...user };
        delete userResponse.password;

        
        res.cookie('auth_token', token, {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'strict',
            maxAge: 7 * 24 * 60 * 60 * 1000 
        });

        res.json({
            success: true,
            message: 'Login successful',
            data: {
                user: userResponse,
                token
            }
        });
    } catch (error) {
        console.error('Login error:', error);
        res.status(500).json({
            success: false,
            message: 'Error logging in',
            error: error.message
        });
    }
};


const logout = async (req, res) => {
    try {
        res.clearCookie('auth_token');
        res.json({
            success: true,
            message: 'Logout successful'
        });
    } catch (error) {
        console.error('Logout error:', error);
        res.status(500).json({
            success: false,
            message: 'Error logging out',
            error: error.message
        });
    }
};


const verifyAuth = async (req, res) => {
    try {
        
        const user = req.user;
        delete user.password;

        res.json({
            success: true,
            data: user
        });
    } catch (error) {
        console.error('Verify auth error:', error);
        res.status(500).json({
            success: false,
            message: 'Error verifying authentication',
            error: error.message
        });
    }
};


const getProfile = async (req, res) => {
    try {
        const db = getDB();
        const user = await db.collection('users').findOne({ id: req.user.userId });

        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        delete user.password;

        res.json({
            success: true,
            data: user
        });
    } catch (error) {
        console.error('Get profile error:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching profile',
            error: error.message
        });
    }
};


const updateProfile = async (req, res) => {
    try {
        const { name, email } = req.body;
        const db = getDB();
        const usersColl = db.collection('users');

        const user = await usersColl.findOne({ id: req.user.userId });
        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        
        if (email && email !== user.email) {
            const existingUsers = await usersColl.find({ email }).toArray();
            if (existingUsers.length > 0) {
                
                const otherUser = existingUsers.find(u => u.id !== req.user.userId);
                if (otherUser) {
                    return res.status(409).json({
                        success: false,
                        message: 'Email already in use'
                    });
                }
            }
        }

        
        const updateFields = {};
        if (name) updateFields.name = name;
        if (email) updateFields.email = email;

        if (Object.keys(updateFields).length > 0) {
            await usersColl.updateOne(
                { id: req.user.userId },
                { $set: updateFields }
            );
        }

        
        const updatedUser = await usersColl.findOne({ id: req.user.userId });
        delete updatedUser.password;

        res.json({
            success: true,
            message: 'Profile updated successfully',
            data: updatedUser
        });
    } catch (error) {
        console.error('Update profile error:', error);
        res.status(500).json({
            success: false,
            message: 'Error updating profile',
            error: error.message
        });
    }
};


const changePassword = async (req, res) => {
    try {
        const { currentPassword, newPassword } = req.body;

        if (!currentPassword || !newPassword) {
            return res.status(400).json({
                success: false,
                message: 'Current password and new password are required'
            });
        }

        const db = getDB();
        const usersColl = db.collection('users');

        const user = await usersColl.findOne({ id: req.user.userId });

        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        
        const isPasswordValid = await bcrypt.compare(currentPassword, user.password);
        if (!isPasswordValid) {
            return res.status(401).json({
                success: false,
                message: 'Current password is incorrect'
            });
        }

        
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(newPassword, salt);

        
        await usersColl.updateOne(
            { id: req.user.userId },
            { $set: { password: hashedPassword } }
        );

        res.json({
            success: true,
            message: 'Password changed successfully'
        });
    } catch (error) {
        console.error('Change password error:', error);
        res.status(500).json({
            success: false,
            message: 'Error changing password',
            error: error.message
        });
    }
};


const getAllUsers = async (req, res) => {
    try {
        const db = getDB();
        const users = await db.collection('users').find({}).toArray();

        
        const sanitizedUsers = users.map(user => {
            const { password, ...rest } = user;
            return rest;
        });

        res.json({
            success: true,
            data: sanitizedUsers
        });
    } catch (error) {
        console.error('Get all users error:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching users',
            error: error.message
        });
    }
};


const updateAdminUser = async (req, res) => {
    try {
        const { userId } = req.params;
        const { name, email, role, isActive, password } = req.body;
        const db = getDB();
        const usersColl = db.collection('users');

        const user = await usersColl.findOne({ id: userId });
        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        const updateFields = {};
        if (name) updateFields.name = name;
        if (email) updateFields.email = email;
        if (role) updateFields.role = role;
        if (isActive !== undefined) updateFields.isActive = isActive;

        if (password) {
            const salt = await bcrypt.genSalt(10);
            updateFields.password = await bcrypt.hash(password, salt);
        }

        await usersColl.updateOne(
            { id: userId },
            { $set: updateFields }
        );

        res.json({
            success: true,
            message: 'User updated successfully'
        });
    } catch (error) {
        console.error('Update admin user error:', error);
        res.status(500).json({
            success: false,
            message: 'Error updating user',
            error: error.message
        });
    }
};


const deleteUser = async (req, res) => {
    try {
        const { userId } = req.params;
        const db = getDB();
        const usersColl = db.collection('users');

        
        if (userId === req.user.userId) {
            return res.status(400).json({
                success: false,
                message: 'You cannot delete your own account'
            });
        }

        await usersColl.deleteOne({ id: userId });

        res.json({
            success: true,
            message: 'User deleted successfully'
        });
    } catch (error) {
        console.error('Delete user error:', error);
        res.status(500).json({
            success: false,
            message: 'Error deleting user',
            error: error.message
        });
    }
};

module.exports = {
    register,
    login,
    logout,
    verifyAuth,
    getProfile,
    updateProfile,
    changePassword,
    getAllUsers,
    updateAdminUser,
    deleteUser
};

