const express = require('express');
const router = express.Router();
const { authMiddleware, authorize } = require('../middleware/authMiddleware');
const { getDB } = require('../config/db');



router.get('/', async (req, res) => {
    try {
        const { category, type, status, limit, search } = req.query;
        const db = getDB();

        const filter = {};
        if (category) filter.category = category;
        if (type) filter.type = type;
        if (search) {
            filter.title = { $regex: search, $options: 'i' };
        }



        if (!status) {
            filter.status = 'published';
        } else if (status !== 'all') {
            filter.status = status;
        }

        let query = db.collection('news').find(filter).sort({ created_at: -1 });


        const total = await db.collection('news').countDocuments(filter);

        const page = parseInt(req.query.page) || 1;
        const pageSize = parseInt(req.query.limit) || 0;
        const skip = (page - 1) * pageSize;

        if (pageSize > 0) {
            query = query.skip(skip).limit(pageSize);
        }

        const news = await query.toArray();


        const mappedNews = news.map(article => ({
            id: article._id.toString(),
            ...article
        }));

        res.json({
            success: true,
            data: mappedNews,
            pagination: {
                total,
                page,
                limit: pageSize,
                totalPages: pageSize > 0 ? Math.ceil(total / pageSize) : 1
            }
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error fetching news',
            error: error.message
        });
    }
});


router.get('/stats', async (req, res) => {
    try {
        const db = getDB();
        const collection = db.collection('news');
        let stats = [];

        if (collection.pool) {

            const [rows] = await collection.pool.query(`
                SELECT JSON_UNQUOTE(JSON_EXTRACT(data, '$.category')) as _id, COUNT(*) as count 
                FROM news 
                GROUP BY JSON_UNQUOTE(JSON_EXTRACT(data, '$.category'))
            `);
            stats = rows;
        } else if (typeof collection.aggregate === 'function') {

            stats = await collection.aggregate([
                {
                    $group: {
                        _id: "$category",
                        count: { $sum: 1 }
                    }
                }
            ]).toArray();
        } else {

            const allNews = await collection.find({}).toArray();
            const counts = {};
            allNews.forEach(n => {
                const cat = n.category || 'Uncategorized';
                counts[cat] = (counts[cat] || 0) + 1;
            });
            stats = Object.entries(counts).map(([k, v]) => ({ _id: k, count: v }));
        }


        const formattedStats = stats.reduce((acc, curr) => {
            if (curr._id) {
                acc[curr._id] = curr.count;
            }
            return acc;
        }, {});

        res.json({
            success: true,
            data: formattedStats
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error fetching news stats',
            error: error.message
        });
    }
});


router.get('/:slug', async (req, res) => {
    try {
        const db = getDB();
        const article = await db.collection('news').findOne({ slug: req.params.slug });

        if (!article) return res.status(404).json({ success: false, message: 'Article not found' });


        const mappedArticle = {
            id: article._id.toString(),
            ...article
        };

        res.json({ success: true, data: mappedArticle });
    } catch (error) {
        res.status(500).json({ success: false, message: 'Error fetching article', error: error.message });
    }
});


router.post('/', authMiddleware, async (req, res) => {
    try {
        const {
            title,
            content,
            category,
            type = 'standard',
            videoUrl = '',
            featuredImage = '',
            status = 'published',
            excerpt = ''
        } = req.body;

        if (!title || !content || !category) {
            return res.status(400).json({
                success: false,
                message: 'Title, content, and category are required'
            });
        }

        const slug = title.toLowerCase()
            .replace(/ /g, '-')
            .replace(/[^\w-]+/g, '') + '-' + Date.now().toString().slice(-4);

        const db = getDB();
        const newArticle = {
            title,
            slug,
            content,
            excerpt: excerpt || content.substring(0, 150).replace(/<[^>]*>/g, '') + '...',
            category,
            type,
            videoUrl,
            featuredImage,
            status,
            hashtags: req.body.hashtags || '',
            isHeadline: req.body.isHeadline || false,
            isSectionFeatured: req.body.isSectionFeatured || false,
            featured: (req.body.isHeadline || req.body.isSectionFeatured) ? true : (req.body.featured || false),
            author: req.body.author || req.user.name,
            authorId: req.user.userId,
            views: 0
        };

        const result = await db.collection('news').insertOne(newArticle);

        // Broadcast to newsletter subscribers
        const { broadcastNewArticle } = require('../utils/emailService');
        broadcastNewArticle({ ...newArticle, id: result.insertedId }).catch(err => {
            console.error('Failed to broadcast newsletter:', err);
        });

        res.status(201).json({
            success: true,
            message: 'Article posted successfully',
            data: { id: result.insertedId, ...newArticle }
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            message: 'Error creating article',
            error: error.message
        });
    }
});





router.put('/:id', authMiddleware, async (req, res) => {
    try {
        const { id } = req.params;
        const db = getDB();

        if (!id) {
            return res.status(400).json({ success: false, message: 'Invalid ID' });
        }

        const query = { id };
        const article = await db.collection('news').findOne(query);

        if (!article) return res.status(404).json({ success: false, message: 'Article not found' });

        const updateData = { ...req.body };

        // Auto-set featured if either big tile flag is true
        if (updateData.isHeadline !== undefined || updateData.isSectionFeatured !== undefined) {
            const isHeadline = updateData.isHeadline !== undefined ? updateData.isHeadline : article.isHeadline;
            const isSectionFeatured = updateData.isSectionFeatured !== undefined ? updateData.isSectionFeatured : article.isSectionFeatured;

            if (isHeadline || isSectionFeatured) {
                updateData.featured = true;
            }
            // Optional: If both are false, do we force featured to false? 
            // The user requirement is "if user turn on any toggle ... that news must show in trending"
            // It doesn't strictly say the reverse. But usually if I untoggle them, I might want to remove it from trending if I didn't explicitly set it.
            // However, since I removed the explicit "Featured" toggle from the UI, 
            // it implies that `featured` is now fully derived from these two.
            else {
                updateData.featured = false;
            }
        }

        delete updateData.id;
        delete updateData._id;





        await db.collection('news').updateOne(query, { $set: updateData });

        // If status is changed to published, broadcast it
        if (updateData.status === 'published' && article.status !== 'published') {
            const { broadcastNewArticle } = require('../utils/emailService');
            broadcastNewArticle({ ...article, ...updateData }).catch(err => {
                console.error('Failed to broadcast newsletter:', err);
            });
        }

        res.json({
            success: true,
            message: 'Article updated successfully'
        });
    } catch (error) {
        res.status(500).json({ success: false, message: 'Error updating article', error: error.message });
    }
});


router.delete('/:id', authMiddleware, async (req, res) => {
    try {
        const { id } = req.params;
        const db = getDB();

        if (!id) {
            return res.status(400).json({ success: false, message: 'Invalid ID' });
        }

        const result = await db.collection('news').deleteOne({ id });

        if (result.deletedCount === 0) return res.status(404).json({ success: false, message: 'Article not found' });

        res.json({ success: true, message: 'Article deleted successfully' });
    } catch (error) {
        res.status(500).json({ success: false, message: 'Error deleting article', error: error.message });
    }
});

module.exports = router;
