const bcrypt = require('bcryptjs');
const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '../.env') });
const { connectToMySQL, getDB } = require('../config/db');

function formatDateForMySQL(date) {
    return date.toISOString().slice(0, 19).replace('T', ' ');
}

async function seed() {
    try {
        console.log('🌱 Starting database seeding...');
        const args = process.argv.slice(2);
        const shouldClean = args.includes('--clean');
        
        
        await connectToMySQL();
        const db = getDB();

        if (shouldClean) {
            console.log('🧹 Cleaning existing data...');
            
            
            
            await db.collection('users').deleteMany({}); 
            await db.collection('gold_silver_rates').deleteMany({});
            await db.collection('news').deleteMany({});
            await db.collection('breaking_news').deleteMany({});
            console.log('✅ Data cleaned.');
        }

        
        console.log('👤 Seeding Users...');
        const email = 'rustamchupa473@gmail.com';
        const password = 'Deepakrajput@321';
        
        const existingUsers = await db.collection('users').find({ email }).toArray();
        if (existingUsers.length === 0) {
            const salt = await bcrypt.genSalt(10);
            const hashedPassword = await bcrypt.hash(password, salt);
            
            await db.collection('users').insertOne({
                email,
                password: hashedPassword,
                name: 'Super Admin',
                role: 'super-admin',
                isActive: true
            });
            console.log('✅ Super Admin created.');
        } else {
            console.log('ℹ️ Super Admin already exists.');
        }

        
        console.log('💰 Seeding Gold/Silver Rates...');
        const existingRates = await db.collection('gold_silver_rates').countDocuments({});
        
        if (existingRates < 5) { 
            const rates = [];
            const today = new Date();
            
            
            for (let i = 0; i < 15; i++) {
                const date = new Date(today);
                date.setDate(date.getDate() - i);
                const dateStr = date.toISOString().split('T')[0]; 

                
                
                const goldPrice = 150000 + Math.floor(Math.random() * 5000) - 2500;
                
                const goldChange = Math.floor(Math.random() * 1000) - 500;
                
                rates.push({
                    type: 'Gold',
                    price: goldPrice.toLocaleString(),
                    unit: 'per tola',
                    change: goldChange > 0 ? `+${goldChange}` : `${goldChange}`,
                    date: dateStr
                });

                
                
                const silverPrice = 1800 + Math.floor(Math.random() * 100) - 50;
                const silverChange = Math.floor(Math.random() * 50) - 25;

                rates.push({
                    type: 'Silver',
                    price: silverPrice.toLocaleString(),
                    unit: 'per tola',
                    change: silverChange > 0 ? `+${silverChange}` : `${silverChange}`,
                    date: dateStr
                });
            }

            for (const rate of rates) {
                
                const exists = await db.collection('gold_silver_rates').findOne({ 
                    type: rate.type, 
                    
                    
                    date: rate.date 
                });
                
                if (!exists) {
                    await db.collection('gold_silver_rates').insertOne(rate);
                }
            }
            console.log(`✅ Seeded ${rates.length} Gold/Silver rates.`);
        } else {
            console.log('ℹ️ Gold/Silver rates already seeded.');
        }

        
        console.log('📂 Seeding Categories...');
        
        const categories = [
            { name: "Business", slug: "business", order: 1 },
            { name: "Politics", slug: "politics", order: 2 },
            { name: "Society", slug: "society", order: 3 },
            { name: "Technology", slug: "technology", order: 4 },
            { name: "Entertainment", slug: "entertainment", order: 5 },
            { name: "Sports", slug: "sports", order: 6 },
            { name: "Health", slug: "health", order: 7 },
            { name: "International", slug: "international", order: 8 }
        ];

        for (const cat of categories) {
            const exists = await db.collection('categories').findOne({ slug: cat.slug });
            if (!exists) {
                await db.collection('categories').insertOne({
                    ...cat,
                    isActive: true
                });
            }
        }
        console.log(`✅ Seeded ${categories.length} Categories.`);

        
        console.log('📰 Seeding News Articles...');
        
        
        const generateArticles = (category, count) => {
            const articles = [];
            const images = {
                Business: "https://images.unsplash.com/photo-1460925895917-afdab827c52f?auto=format&fit=crop&q=80&w=1000",
                Politics: "https://images.unsplash.com/photo-1529101091760-61df6be24296?auto=format&fit=crop&q=80&w=1000",
                Society: "https://images.unsplash.com/photo-1455849318743-b2233052fcff?auto=format&fit=crop&q=80&w=1000",
                Technology: "https://images.unsplash.com/photo-1488590528505-98d2b5aba04b?auto=format&fit=crop&q=80&w=1000",
                Entertainment: "https://images.unsplash.com/photo-1499364615650-ec38552f4f34?auto=format&fit=crop&q=80&w=1000",
                Sports: "https://images.unsplash.com/photo-1461896836934-ffe607ba8211?auto=format&fit=crop&q=80&w=1000",
                Health: "https://images.unsplash.com/photo-1505751172876-fa1923c5c528?auto=format&fit=crop&q=80&w=1000",
                International: "https://images.unsplash.com/photo-1526778548025-fa2f459cd5c1?auto=format&fit=crop&q=80&w=1000"
            };

            for (let i = 1; i <= count; i++) {
                const date = new Date();
                date.setDate(date.getDate() - Math.floor(Math.random() * 10)); 
                
                articles.push({
                    title: `${category} News Update #${i}: Major Developments in 2026`,
                    slug: `${category.toLowerCase()}-news-update-${i}-2026`,
                    content: `<p>This is a detailed report about the latest happenings in the <strong>${category}</strong> sector. Analysts predict significant changes in the coming weeks.</p><p>Stay tuned for more updates as the situation evolves.</p>`,
                    excerpt: `Latest updates and analysis from the ${category} world. Read more to find out what's happening.`,
                    image: images[category] || images.Business,
                    category: category,
                    author: "Super Admin",
                    status: "published",
                    published_at: formatDateForMySQL(date)
                });
            }
            return articles;
        };

        let allArticles = [];
        for (const cat of categories) {
            const catArticles = generateArticles(cat.name, 5);
            allArticles = [...allArticles, ...catArticles];
        }

        
        

        let seededCount = 0;
        for (const article of allArticles) {
            
            const exists = await db.collection('news').findOne({ slug: article.slug });
            if (!exists) {
                await db.collection('news').insertOne(article);
                seededCount++;
            }
        }
        console.log(`✅ Seeded ${seededCount} News articles across ${categories.length} categories.`);

        
        console.log('🚨 Seeding Breaking News...');
        const existingBreaking = await db.collection('breaking_news').countDocuments({});
        
        if (existingBreaking < 3) {
            const breakingNews = [
                {
                    title: "Heavy Rainfall Expected in Kathmandu Valley Tomorrow",
                    link: "",
                    status: "active"
                },
                {
                    title: "Prime Minister to Address the Nation at 8 PM",
                    link: "",
                    status: "active"
                },
                {
                    title: "Major Traffic Jam at Ring Road Due to Maintenance",
                    link: "",
                    status: "active"
                },
                {
                    title: "Gold Prices Drop Sharply in International Market",
                    link: "/gold-silver",
                    status: "active"
                }
            ];

            for (const news of breakingNews) {
                const exists = await db.collection('breaking_news').findOne({ title: news.title });
                if (!exists) {
                    await db.collection('breaking_news').insertOne(news);
                }
            }
            console.log(`✅ Seeded ${breakingNews.length} Breaking News items.`);
        } else {
            console.log('ℹ️ Breaking News already seeded.');
        }

        
        console.log('💼 Seeding Careers...');
        const jobs = [
            {
                title: "Senior News Editor",
                department: "Editorial",
                location: "Kathmandu, Nepal",
                type: "Full-time",
                description: "We are looking for an experienced Senior News Editor to lead our editorial team. The ideal candidate will have a strong background in journalism and editorial leadership.",
                requirements: [
                    "Bachelor's degree in Journalism or related field",
                    "5+ years of experience in news editing",
                    "Strong leadership skills",
                    "Excellent command of English and Nepali"
                ],
                deadline: "2026-03-01",
                status: "active",
                posted_at: new Date().toISOString()
            },
            {
                title: "Investigative Journalist",
                department: "Editorial",
                location: "Kathmandu, Nepal",
                type: "Full-time",
                description: "Join our investigative team to uncover stories that matter. You will work on in-depth reports and special features.",
                requirements: [
                    "Proven experience as an investigative journalist",
                    "Strong research and analytical skills",
                    "Ability to work under pressure",
                    "Willingness to travel"
                ],
                deadline: "2026-03-15",
                status: "active",
                posted_at: new Date().toISOString()
            },
            {
                title: "Digital Marketing Specialist",
                department: "Marketing",
                location: "Remote / Hybrid",
                type: "Contract",
                description: "We need a Digital Marketing Specialist to grow our online presence and manage social media campaigns.",
                requirements: [
                    "Experience with SEO/SEM and social media marketing",
                    "Data-driven mindset",
                    "Creativity in content strategy",
                    "Knowledge of analytics tools"
                ],
                deadline: "2026-02-28",
                status: "active",
                posted_at: new Date().toISOString()
            }
        ];

        let seededJobsCount = 0;
        for (const job of jobs) {
            const exists = await db.collection('careers').findOne({ title: job.title });
            if (!exists) {
                await db.collection('careers').insertOne(job);
                seededJobsCount++;
            }
        }
        console.log(`✅ Seeded ${seededJobsCount} Careers.`);

        console.log('✨ Database seeding completed successfully!');
        process.exit(0);

    } catch (error) {
        console.error('❌ Error seeding database:', error);
        process.exit(1);
    }
}

seed();
